import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Star, Package, CheckCircle, Award, Apple, TrendingUp, Heart, Target, Zap, Users, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyConsultations.css'

const MyConsultations = () => {
  const navigate = useNavigate()
  const [consultations, setConsultations] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('nutriConsultations')
    if (saved) {
      try {
        setConsultations(JSON.parse(saved))
      } catch (e) {
        setConsultations([])
      }
    } else {
      const defaultConsultations = [
        {
          id: 1,
          plan: 'Сбалансированное питание',
          date: '2025-01-20',
          rating: 5
        },
        {
          id: 2,
          plan: 'Детокс программа',
          date: '2025-01-10',
          rating: 4
        },
        {
          id: 3,
          plan: 'План для набора массы',
          date: '2024-12-28',
          rating: 5
        }
      ]
      setConsultations(defaultConsultations)
      localStorage.setItem('nutriConsultations', JSON.stringify(defaultConsultations))
    }
  }, [])

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  return (
    <div className="my-consultations-page-nutrition">
      <div className="container">
        <section className="page-header-nutrition">
          <div className="header-badge-nutrition">
            <Apple size={20} />
            <span>ВАШИ КОНСУЛЬТАЦИИ</span>
          </div>
          <h1 className="page-title-nutrition">
            Мои <span className="title-accent-nutrition">консультации</span>
          </h1>
          <p className="page-description-nutrition">
            Отслеживайте свои планы питания и результаты консультаций
          </p>
        </section>

        <ImageSlider />

        <FloatingElementsSection />

        <NutritionTipsSection />

        {consultations.length === 0 ? (
          <div className="empty-state-nutrition">
            <div className="empty-icon-wrapper-nutrition">
              <Package size={64} />
            </div>
            <h2 className="empty-title-nutrition">Нет консультаций</h2>
            <p className="empty-text-nutrition">Запишитесь на первую консультацию с нутрициологом</p>
            <button 
              className="empty-button-nutrition" 
              onClick={() => navigate('/enroll')}
            >
              Записаться на консультацию
            </button>
          </div>
        ) : (
          <section className="consultations-list-nutrition">
            <div className="consultations-section-header-nutrition">
              <h2 className="consultations-section-title-nutrition">Мои записи на консультации</h2>
              <p className="consultations-section-description-nutrition">
                Все ваши планы питания и результаты работы с нутрициологом
              </p>
            </div>
            <div className="consultations-grid-nutrition">
              {consultations.map((consultation) => (
                <ConsultationCard
                  key={consultation.id}
                  consultation={consultation}
                  formatDate={formatDate}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const ConsultationCard = ({ consultation, formatDate }) => {
  return (
    <div className="consultation-item-nutrition">
      <div className="consultation-header-nutrition">
        <div className="consultation-icon-wrapper-nutrition">
          <Apple size={32} />
        </div>
        <div className="consultation-info-nutrition">
          <div className="consultation-number-nutrition">#{consultation.id}</div>
          <div className="rating-badge-nutrition">
            <Star size={16} fill="#FFB74D" color="#FFB74D" />
            <span>{consultation.rating}</span>
          </div>
        </div>
      </div>
      
      <h3 className="consultation-title-nutrition">{consultation.plan}</h3>
      
      <div className="consultation-details-nutrition">
        <div className="detail-item-nutrition">
          <Calendar size={18} />
          <div>
            <span className="detail-label-nutrition">Дата консультации:</span>
            <span className="detail-value-nutrition">{formatDate(consultation.date)}</span>
          </div>
        </div>
        <div className="detail-item-nutrition">
          <Star size={18} />
          <div>
            <span className="detail-label-nutrition">Оценка результатов:</span>
            <div className="rating-stars-nutrition">
              {Array.from({ length: 5 }).map((_, i) => (
                <Star
                  key={i}
                  size={16}
                  fill={i < consultation.rating ? "#FFB74D" : "none"}
                  color="#FFB74D"
                />
              ))}
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

const FloatingElementsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const nutrients = [
    { name: 'Белки', value: 30, color: '#4CAF50', icon: '🥩' },
    { name: 'Углеводы', value: 40, color: '#FF9800', icon: '🌾' },
    { name: 'Жиры', value: 30, color: '#FF6B6B', icon: '🥑' }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`floating-elements-section-nutrition ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="floating-elements-header-nutrition">
          <h2 className="floating-elements-title-nutrition">Баланс макронутриентов</h2>
          <p className="floating-elements-description-nutrition">
            Оптимальное соотношение для здорового питания
          </p>
        </div>
        <div className="floating-elements-grid-nutrition">
          {nutrients.map((nutrient, index) => (
            <div 
              key={index} 
              className="floating-nutrient-card-nutrition"
              style={{ 
                animationDelay: `${index * 0.2}s`,
                '--nutrient-color': nutrient.color
              }}
            >
              <div className="floating-nutrient-emoji-nutrition">{nutrient.icon}</div>
              <div className="floating-nutrient-circle-nutrition">
                <svg className="floating-nutrient-svg-nutrition" viewBox="0 0 120 120">
                  <circle
                    className="floating-nutrient-bg-nutrition"
                    cx="60"
                    cy="60"
                    r="50"
                  />
                  <circle
                    className="floating-nutrient-progress-nutrition"
                    cx="60"
                    cy="60"
                    r="50"
                    style={{
                      strokeDasharray: `${2 * Math.PI * 50}`,
                      strokeDashoffset: `${2 * Math.PI * 50 * (1 - nutrient.value / 100)}`
                    }}
                  />
                </svg>
                <div className="floating-nutrient-value-nutrition">{nutrient.value}%</div>
              </div>
              <h3 className="floating-nutrient-name-nutrition">{nutrient.name}</h3>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const NutritionTipsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const tips = [
    {
      icon: Heart,
      title: 'Пейте больше воды',
      description: 'Не менее 2 литров чистой воды в день для поддержания метаболизма',
      color: 'var(--nutrition-coral)'
    },
    {
      icon: Apple,
      title: 'Ешьте регулярно',
      description: '5-6 небольших приемов пищи в день лучше, чем 2-3 больших',
      color: 'var(--nutrition-green)'
    },
    {
      icon: Target,
      title: 'Следите за балансом',
      description: 'Белки, жиры и углеводы должны быть сбалансированы в каждом приеме пищи',
      color: 'var(--nutrition-orange)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`nutrition-tips-section-nutrition ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="nutrition-tips-header-nutrition">
          <h2 className="nutrition-tips-title-nutrition">Советы по питанию</h2>
          <p className="nutrition-tips-description-nutrition">
            Простые правила для здорового образа жизни
          </p>
        </div>
        <div className="nutrition-tips-grid-nutrition">
          {tips.map((tip, index) => {
            const Icon = tip.icon
            return (
              <div 
                key={index} 
                className="nutrition-tip-card-nutrition"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--tip-color': tip.color
                }}
              >
                <div className="nutrition-tip-icon-wrapper-nutrition">
                  <Icon size={40} />
                </div>
                <h3 className="nutrition-tip-title-nutrition">{tip.title}</h3>
                <p className="nutrition-tip-text-nutrition">{tip.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default MyConsultations

